/* FILE: mlocate.c                              (D. Tottingham  12/26/90)

This is a collection of C helper functions that manage event location for
xdetect.  All functions have been written and compiled medium model.  The
following functions are included:

l_display_location ()           write location to stream
l_get_first_location ()         get first location
l_get_location ()               get location if event was located
l_get_location_status ()        get location_enabled
l_get_next_location ()          get next location
l_initialize ()                 initialize location module
l_locate_event ()               locate the event
l_reset_location ()             reset location flag
l_set_location_status ()        set location_enabled flag
l_toggle_location_status ()     toggle location_enabled flag

EXTERNAL FUNCTIONS CALLED:

er_abort ()                     display an error message then quit
hs_initialize ()                initialize halfspace module
hs_locate_event ()              locate event using halfspace algorithm
q_dequeue ()                    dequeue a data link from a data queue
q_enqueue ()                    enqueue a data link on a data queue
q_initialize ()                 initialize a data queue
suds_initialize ()              initialize a suds structure
suds_initialize_tag ()          initialize a suds structtag

HISTORY:

V1.95   (09/29/90)  Added rms residual to l_display_location().

*/



/*************************************************************************
                            INCLUDE FILES

*************************************************************************/
#include <malloc.h>
#include <math.h>
#include <stdio.h>

#include "mconst.h"
#include "merror.h"
#include "mfile.h"
#include "mhalfsp.h"
#include "mlocate.h"
#include "mqueue.h"
#include "msudsini.h"
#include "mutils.h"


/*************************************************************************
                              GLOBAL DATA

**************************************************************************/
PRIVATE Q_QUEUE location_queue;
PRIVATE Q_LINK * head_ptr;
PRIVATE unsigned int location_count;
PRIVATE FLAG location_flag;
PRIVATE FLAG location_enabled;


/*=======================================================================*
 *                            adjust_queue                               *
 *=======================================================================*/
/* Create and save a data link in a data queue.                          */

PRIVATE
void adjust_queue (location)
Q_LOCATION far * location;
{
   Q_TYPE type;

   type.location = location;
   q_enqueue (&location_queue, type);
   if (location_count == MAX_LOCATIONS) {
      if (q_dequeue (&location_queue, &type))
         _ffree (type.location);
   } else location_count++;
}

/*=======================================================================*
 *                          l_display_location                           *
 *=======================================================================*/
/* Write location to stream.                                             */

PUBLIC
void l_display_location (stream)
FILE * stream;
{
   Q_LOCATION far * location;
   char * timeline;

   if (location_flag) {
      location = location_queue.tail->type.location;

      fprintf (stream, "DATE        ORIGIN        LAT       LONG       DEPTH   MAG  NO RMS  ERH  ERZ QM\n");
      timeline = u_build_timeline (location->info.orgtime, 2);
      fprintf (stream, "%s  %6.3lf %c  %7.3lf %c  %7.2lf ",
                timeline, fabs(location->info.or_lat),
                (location->info.or_lat >= 0.0) ? 'N' : 'S',
                fabs(location->info.or_long),
                (location->info.or_long >= 0.0) ? 'E' : 'W',
                location->info.depth);
      if (location->info.magnitude != atof(NODATA))
         fprintf (stream, "%4.1f", location->info.magnitude);
      else fprintf (stream, "    ");

      fprintf (stream, "%2d %4.1f\n", location->info.num_stats,
                location->info.res_rms);

      free (timeline);
   } else fprintf (stream, "\n*** Unable to locate event ***\n");
}

/*=======================================================================*
 *                        l_get_first_location                           *
 *=======================================================================*/
/* Get first location from location queue.                               */

PUBLIC
Q_LOCATION far * l_get_first_location ()
{
   head_ptr = location_queue.head;
   if (head_ptr != NULL)
      return (head_ptr->type.location);
   else return (NULL);
}

/*=======================================================================*
 *                            l_get_location                             *
 *=======================================================================*/
/* Get location if event was located.                                    */

PUBLIC
Q_LOCATION far * l_get_location ()
{
   if (location_flag && location_queue.tail != NULL)
      return (location_queue.tail->type.location);
   else return (NULL);
}

/*=======================================================================*
 *                         l_get_location_status                         *
 *=======================================================================*/
/* Get location_enabled.                                                 */

PUBLIC
FLAG l_get_location_status ()
{
   return location_enabled;
}

/*=======================================================================*
 *                         l_get_next_location                           *
 *=======================================================================*/
/* Get next location from location queue.                                */

PUBLIC
Q_LOCATION far * l_get_next_location ()
{
   head_ptr = head_ptr->next;
   if (head_ptr != NULL)
      return (head_ptr->type.location);
   else return (NULL);
}

/*=======================================================================*
 *                             l_initialize                              *
 *=======================================================================*/
/* Initialize the location module.                                       */

PUBLIC
void l_initialize ()
{
   location_enabled = LOCATION_ENABLED;
   q_initialize (&location_queue);
   location_count = 0;
   location_flag = FALSE;
   hs_initialize ();
}

/*=======================================================================*
 *                            l_locate_event                             *
 *=======================================================================*/
/* Locate event using halfspace algorithm.                               */

PUBLIC
FLAG l_locate_event ()
{
   Q_LOCATION far *location_ptr;
   Q_TYPE type;

   if (! location_enabled)
      return FALSE;

   /* Initialize a new origin structure */
   location_ptr = (Q_LOCATION far *) _fmalloc (sizeof(Q_LOCATION));
   if (location_ptr == NULL) er_abort (L_NO_STORAGE);
   suds_initialize_tag (ORIGIN, &location_ptr->structtag);
   suds_initialize (ORIGIN, &location_ptr->info);

   /* Do a halfspace location */
   if (location_flag = hs_locate_event (&location_ptr->info)) {
      location_ptr->info.number = f_get_event_number();
      location_ptr->info.authority = st_get_authority();
      location_ptr->info.version = '0';
      location_ptr->info.or_status = 'a';
      location_ptr->info.program = 's';
      location_ptr->info.effective = u_timestamp();
      adjust_queue (location_ptr);
   } else _ffree (location_ptr);

   return (location_flag);
}

/*=======================================================================*
 *                           l_reset_location                            *
 *=======================================================================*/
/* Reset the location flag.                                              */

PUBLIC
void l_reset_location ()
{
   location_flag = FALSE;
}

/*=======================================================================*
 *                       l_set_location_status                           *
 *=======================================================================*/
/* Set location_enabled flag.                                            */

PUBLIC
void l_set_location_status (location_status)
FLAG location_status;
{
   location_enabled = location_status;
}

/*=======================================================================*
 *                      l_toggle_location_status                         *
 *=======================================================================*/
/* Toggle location_enabled flag.                                         */

PUBLIC
void l_toggle_location_status ()
{
   location_enabled = (location_enabled) ? FALSE : TRUE;
}
